import { useState, useRef, useEffect } from "react";
import axios from "axios";
import { API_URL } from "../../constants";
import { bearerHeader } from "../../auth";
import type { ApiResponseProps } from "../../types/app";
import toast from "react-hot-toast";

interface FormUploadTextProps {
  type: string;
  className?: string;
  onSuccess?: (response: ApiResponseProps) => void;
  apiPath?: string;
}

const FormUploadText: React.FC<FormUploadTextProps> = ({ type, className, onSuccess, apiPath }) => {
  const [uploading, setUploading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState<boolean>(false);

  const fileInputRef = useRef<HTMLInputElement | null>(null);

  const fileAccept = (): string => {
    switch (type) {
      case "image":
        return "image/*";
      case "video":
        return "video/*";
      case "document":
        return ".pdf,.doc,.docx,.txt,.xls,.xlsx";
      default:
        return "*/*";
    }
  };

  const handleFileUpload = async (selectedFile: File) => {
    setUploading(true);
    setError(null);

    const formData = new FormData();
    formData.append("file", selectedFile);

    try {
      const PATH = apiPath ? apiPath : `upload/${type}`;
      const response = await axios.post(API_URL + PATH, formData, {
        headers: {
          "Content-Type": "multipart/form-data",
          ...bearerHeader(),
        },
      });

      setSuccess(true);
      if (onSuccess) {
        onSuccess(response.data);
      }
    } catch (err: any) {
      setError(err.response?.data?.message || "File upload failed");
    } finally {
      setUploading(false);

      // Reset the input value to allow re-uploading the same file
      if (fileInputRef.current) {
        fileInputRef.current.value = "";
      }
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      handleFileUpload(e.target.files[0]);
    }
  };

  useEffect(() => {
    if (success) {
      toast.success("File uploaded successfully!");
    }
    if (error) {
      toast.error(error);
    }
  }, [success, error]);

  return (
    <div 
      className="upload-container"
      onClick={(e) => {
        e.stopPropagation(); // Prevent click event propagation
        fileInputRef.current?.click();
      }}
    >
      <label
        className="peer-disabled:cursor-not-allowed peer-disabled:opacity-70 flex justify-center p-2 gap-1 cursor-pointer mt-1 bg-[#f4f2f3] rounded-xl items-center text-xs font-semibold"
        htmlFor="upload-input"
      >
        <span>+</span>
        {uploading ? (
          <p className="text-blue-500">Uploading...</p>
        ) : (
          <p className="text-gray-600 font-semibold">Upload New</p>
        )}
      </label>
      <input
        className="hidden"
        id="upload-input"
        type="file"
        accept={fileAccept()}
        onChange={handleFileChange}
        ref={fileInputRef}
      />
    </div>
  );
};

export default FormUploadText;
