export function slugToNormal(text: string): string {
    return text
        .split(/[^a-zA-Z0-9]+/)       // Split the string by any non-alphanumeric characters
        .map(word =>
            word.charAt(0).toUpperCase() + word.slice(1).toLowerCase()
        )                             // Capitalize the first letter of each word
        .join(' ');                   // Join the words back into a single string
}


export const isEmpty = (val: any): boolean => {
    return val == null || !(Object.keys(val) || val).length;
};

export const formatText = (text: string): string => text.trim().replace(/ /g, '-').toLowerCase()

export const strShort = (str: string, n: number): string => (str.length > n) ? str.substr(0, n - 1) + '...' : str;

export const strShort2 = (str: string, n: number): string => (str.length > n) ? str.substr(0, n - 1) : str;

export const toStringList = (data: string[], symbol: string = ', '): string => {
    if (isEmpty(data)) return '';
    let a = '';
    data.forEach(x => {
        if (data.length === 1) a += `${x}`;
        else {
            a += `${x}${symbol}`;
        }
    });
    a = a + '.';
    return a.replace(`${symbol}.`, '');
};

export const toUpperCaseFirstString = (string: string): string => {
    if (!string) return '';
    let newString = '';
    const firstCase = string[0];
    newString = firstCase.toUpperCase() + string.slice(1);
    return newString;
};

export const toSlug = (string: string, separator: string = '-'): string => {
    return string.replace(/[^\w\s]/g, separator)
        .replace(/\s+/g, separator)
        .toLowerCase();
};

export const isItAllArabic = (s: string): boolean => {
    return !/[^\u0600-\u06FF\u0020-\u0040\u005B-\u0060\u007B-\u007E]/.test(s);
};


export function ucFirst(str: string): string {
    if (!str) return str; // If the string is empty, return it as is.
    return str.charAt(0).toUpperCase() + str.slice(1);
}


export const randomNumber = (min: number, max: number): number => Math.floor(Math.random() * (max - min + 1)) + min;


// Utility function to convert enum to an array of strings
export function enumToArray<T extends object>(enumObj: T): string[] {
    return Object.values(enumObj).filter(value => typeof value === 'string') as string[];
}


// Utility function to convert enum to a comma-separated string
export function enumToString<T extends Record<string, unknown>>(enumObj: T): string {
    return enumToArray(enumObj).join(',');
}

export function getDateTimeComponents(iso8601String: string): { date: string; time: string } {
    const date = new Date(iso8601String);

    return {
        date: date.toISOString().split('T')[0],    // Extract the date component
        time: date.toISOString().split('T')[1].split('.')[0], // Extract the time component
    };
}

export function ensureHttpsUrl(url: string | undefined, defaultDomain: string): string {
    // Return default domain if URL is undefined or empty
    if (!url) {
        return defaultDomain;
    }

    // Ensure the URL starts with "https://"
    if (!url.startsWith("https://")) {
        // If URL doesn't start with a slash, append a slash to the default domain
        if (!url.startsWith("/")) {
            defaultDomain += "/";
        }
        // Combine default domain with the URL
        url = defaultDomain + url;
    }

    // Remove duplicate slashes
    url = url.replace(/([^:]\/)\/+/g, "$1");

    return url;
}


export const getFormValues = (formName: string): Record<string, string> => {
    const form = document.querySelector<HTMLFormElement>(`#${formName}`);
    if (!form) {
        throw new Error(`Form with name ${formName} not found`);
    }
    const formData = new FormData(form);
    return Object.fromEntries(formData.entries()) as Record<string, string>;
};

export function convertToSubcurrency(amount: number, factor = 100) {
    return Math.round(amount * factor);
}


export function encodeObject(obj: object) {
    const jsonString = JSON.stringify(obj);
    return btoa(jsonString); // Base64 encoding
}

export function decodeObject(encodedString: string) {
    const jsonString = atob(encodedString); // Base64 decoding
    return JSON.parse(jsonString);
}



// Generic function to remove duplicates based on a key
export const removeDuplicates = <T>(arr: T[], key: keyof T): T[] =>
    Array.from(
        arr.reduce((map, obj) => map.set(obj[key], obj), new Map<any, T>()).values()
    );

// Function to remove duplicates from an array of primitive values
export const removeDuplicatesArr = <T>(arr: T[]): T[] =>
    Array.from(
        arr.reduce((map, obj) => map.set(obj, obj), new Map<T, T>()).values()
    );

// Function to create a sorting order based on a property
export const GetSortOrder = <T>(prop: keyof T) => {
    return (a: T, b: T): number => {
        if (a[prop] < b[prop]) return 1;
        else if (a[prop] > b[prop]) return -1;
        return 0;
    };
};


export function groupArrayOfObjectByKey<T, K extends keyof T>(array: T[], key: K): { topic: string; data: T[] }[] {
    const groupedObject = array.reduce((result, currentValue) => {
        const keyValue = currentValue[key] as unknown as string;

        if (!result[keyValue]) {
            result[keyValue] = [];
        }

        result[keyValue].push(currentValue);

        return result;
    }, {} as Record<string, T[]>);

    // Convert the grouped object into an array of objects with { topic, data }
    return Object.entries(groupedObject).map(([topic, data]) => ({ topic, data }));
}

export function getRandomItems<T>(array: T[], size: number): T[] {
    const result: T[] = [];
    const arrayCopy = [...array]; // Create a copy of the array to avoid modifying the original array

    // Limit size to the array length to avoid an infinite loop
    const maxSize = Math.min(size, array.length);

    while (result.length < maxSize) {
        const randomIndex = Math.floor(Math.random() * arrayCopy.length);
        result.push(arrayCopy[randomIndex]);
        arrayCopy.splice(randomIndex, 1); // Remove the selected item to avoid duplicates
    }

    return result;
}

export function exportTableToCSV(tableId: string, filename: string = 'table.csv'): void {
    // Get the table element by ID
    const table = document.getElementById(tableId) as HTMLTableElement | null;
    if (!table) {
        console.error(`Table with ID "${tableId}" not found.`);
        return;
    }

    let csvContent = '';
    for (const row of table.rows) {
        const rowData: string[] = [];
        for (const cell of row.cells) {
            // Escape commas and double-quotes
            const cellContent = cell.textContent ? cell.textContent.replace(/"/g, '""') : '';
            rowData.push(`"${cellContent}"`);
        }
        csvContent += rowData.join(',') + '\n';
    }

    // Create a blob with the CSV content and download the file
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = filename;
    link.style.display = 'none';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}


export const total = (data: number[]) => data.reduce((m, prev) => m + prev, 0)

 /**
* Converts an array of objects into a CSV format and triggers a download.
* @param data - The data to be converted into a CSV file.
* @param fileName - The name of the downloaded CSV file (default is "data.csv").
*/
export const downloadCSV = (data: Record<string, any>[], fileName: string = "data.csv") => {
   if (!data || data.length === 0) {
       console.error("No data provided to generate CSV.");
       return;
   }

   // Extract headers from the first object
   const headers = Object.keys(data[0]);

   // Convert data array to CSV format
   const csvContent = [
       headers.join(","), // Add header row
       ...data.map(row =>
           headers
               .map(header => JSON.stringify(row[header] ?? "")) // Handle null/undefined fields
               .join(",")
       )
   ].join("\n");

   // Create a Blob with CSV content
   const blob = new Blob([csvContent], { type: "text/csv;charset=utf-8;" });

   // Create a download link
   const link = document.createElement("a");
   link.href = URL.createObjectURL(blob);
   link.download = fileName;

   // Trigger the download
   document.body.appendChild(link);
   link.click();
   document.body.removeChild(link);
};
