<?php

namespace App\Http\Controllers\RestApi\V1;

use App\Contracts\OtpSenderInterface;
use App\Enums\UserTypeEnum;
use App\Http\Controllers\RestApiController;
use App\Models\User;
use App\Services\Model\UserService;
use App\Utils\Otp;
use Illuminate\Auth\Events\Registered;
use Illuminate\Auth\Events\Verified;
use Illuminate\Foundation\Auth\EmailVerificationRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Validation\Rules\Password;

class AuthController extends RestApiController
{

    public function __construct(
        protected OtpSenderInterface $otpSender,
        protected UserService $userService,
    ) {
    }

    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|exists:users,name',
            'password' => 'required|string|min:10',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $user = User::where('email', $request->email)->whereNot('email', null)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return $this->errorResponse('Invalid credentials', 401);
        }

        $token = $user->createToken($user->password)->plainTextToken;

        return $this->successResponse(compact('token'),'successfully login');

    }

    public function loginPhone(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string|max:15',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $user = User::where('phone', $request->phone)->first();

        if (!$user) {
            return $this->badRequestResponse('user with this phone not found');
        }

        $token = Str::random(60);

        $user->update(['api_token' => $token]);

        $otpUtil = new Otp();
        $otp = $otpUtil->makeOtp($user->id);

        if ($this->otpSender->send($otp)) {
            return $this->successResponse(['token' => $token], 'OTP sended to your phone');

        }
        return $this->badRequestResponse('otp send failed');

    }
    public function registerStudent(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:users',
            'phone' => 'required|string|max:15|unique:users',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $token = Str::random(60);

        $user = User::create([
            'name' => $request->name,
            'phone' => $request->phone,
            'password' => Hash::make(Str::random(20)),
            'type'=>UserTypeEnum::STUDENT,
            'api_token' => $token
        ]);

        $isCreated = $this->userService->setUser($user)->createStudent();

        if (!$isCreated) {
            throw new \Exception("Student not created");
        }

        $otpUtil = new Otp();
        $otp = $otpUtil->makeOtp($user->id);

        if ($this->otpSender->send($otp)) {
            event(new Registered($user));
            return $this->successResponse(['token' => $token], 'OTP sended to your phone');

        }
        return $this->badRequestResponse('otp send failed');
    }

    public function verifyOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'otp' => 'required|string|size:6',
            'token' => 'required|exists:users,api_token',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        // Check if the token exists in the request and if it's valid
        $user = User::where('api_token', $request->token)->first();

        // Verify the OTP (you need to implement this based on your OTP logic)
        $otpUtil = new Otp();
        $isValidOtp = $otpUtil->checkOtp($user->id, $request->otp);

        if (!$isValidOtp) {
            return $this->validationErrorResponse(['otp' => 'Invalid OTP.']);
        }

        $token = $user->createToken($user->password)->plainTextToken;

        return $this->successResponse(compact('token'),'successfully login');
    }

    public function validateTokenMethod(Request $request)
    {
        $token = $request->bearerToken();

        if (!$token) {
            return $this->errorResponse('Authorization token not provided', 401);
        }

        try {
            $tokenModel = \Laravel\Sanctum\PersonalAccessToken::findToken($token);

            if (!$tokenModel) {
                return $this->errorResponse( 'Invalid or expired token', 401);
            }

            if ($tokenModel->expires_at && $tokenModel->expires_at->isPast()) {
                return $this->errorResponse('Token has expired', 401);
            }

            $user = $tokenModel->tokenable;
            $user['token'] = $token;

            return $user;
        } catch (\Exception $e) {
            return $this->errorResponse( 'An error occurred during token validation: ' . $e->getMessage(), 500);
        }
    }

    public function verifyEmail(EmailVerificationRequest $request)
    {
        if ($request->user()->hasVerifiedEmail()) {
            return $this->errorResponse('email already verified',400);
        }

        if ($request->user()->markEmailAsVerified()) {
            event(new Verified($request->user()));
        }

        return $this->successResponse(null,'email verified successfully');
    }

    public function emailVerificationNotification(Request $request)
    {
        if ($request->user()->hasVerifiedEmail()) {
            return $this->badRequestResponse('email already verified');
        }

        $request->user()->sendEmailVerificationNotification();

        return $this->successResponse('verification link sent');
    }

    public function updatePassword(Request $request)
    {
        $validated = $request->validate([
            'current_password' => ['required', 'current_password'],
            'password' => ['required', Password::defaults(), 'confirmed'],
        ]);

        $request->user()->update([
            'password' => Hash::make($validated['password']),
        ]);

        return $this->successResponse(null,'password updated successfully');
    }
}
