import { FormElement, FormInput, Modal, PrimaryButton, SmallCard } from '@/Components'
import DashboardLayout from '@/Layouts/DashboardLayout'
import { PageProps } from '@/types'
import { ID } from '@/types/models'
import { Permission, Role } from '@/types/models/role'
import { ButtonAdd, ButtonCreate, ButtonDelete, ButtonEdit } from '@/Ui'
import { removeDuplicates } from '@/Utils'
import { DialogTitle } from '@headlessui/react'
import { useForm } from '@inertiajs/react'
import { Divider, Select } from 'antd'
import React, { FormEventHandler, useEffect, useState } from 'react'

type Props = {
    roles: Role[]
    permissions: Permission[]
}

const Roles = ({ roles, permissions }: PageProps<Props>) => {

    const [modalOpen, setModalOpen] = useState<boolean>(false)
    const [modelSection, setModelSection] = useState('')
    const [selectedRole, setSelectedRole] = useState<Role>()
    const [selectedPermissions, setSelectedPermissions] = useState<Permission[]>([])

    const closeModal = () => {
        setModalOpen(false)
        setModelSection('')
    }

    const defaultRole: Role = {
        id: '',
        name: '',
        permissionsIDs: [],
        permissions: []
    }
    const form = useForm(selectedRole ?? defaultRole)

    const handleAdd = () => {
        setModalOpen(true)
        setSelectedRole(undefined)
        form.setData(defaultRole)
        setSelectedPermissions(defaultRole.permissions)
        setModelSection('form')
    }

    const handleUpdate = (item: Role) => {
        setModalOpen(true)
        setSelectedRole(item)
        setSelectedPermissions(item.permissions)
        form.setData(item)
        setModelSection('form')
    }

    const create: FormEventHandler = (e) => {
        e.preventDefault()
        form.post(route('admin.role.createOne'), {
            onSuccess: () => {
                setModalOpen(false)
            }
        })
    }

    const update: FormEventHandler = (e) => {
        e.preventDefault()
        form.put(route('admin.role.updateOne', selectedRole?.id), {
            onSuccess: () => {
                setModalOpen(false)
            }
        })
    }

    const handleSelect = (id: ID) => {

        const item = permissions.find(m => m.id == id)

        if (item) setSelectedPermissions(
            removeDuplicates([...form.data.permissions, item], 'id')
        )
    }

    useEffect(() => {
        form.setData('permissionsIDs',
            selectedPermissions.map(m => m.id)
        )

    }, [selectedPermissions])


    return (
        <DashboardLayout title='الأدوار' headChild={<ButtonAdd onClick={handleAdd} />}>
            <Modal show={modalOpen} onClose={closeModal} >
                {modelSection === 'form' && (
                    <form onSubmit={selectedRole ? update : create} >
                        <FormInput label='الإسم' value={form.data.name} onChange={e => form.setData('name', e.target.value)} />
                        <FormElement label='الصف' >
                            <Select className='w-full' onSelect={handleSelect} >
                                {permissions.map(m => <Select.Option value={m.id} key={m.id} > {m.name} </Select.Option>)}
                            </Select>
                        </FormElement>
                        <div className="flex ">
                            {selectedPermissions?.map((item, i) =>
                                <SmallCard key={i}>
                                    {item.name}
                                    <ButtonDelete onClick={() => setSelectedPermissions(permissions.filter(m => m.id !== item.id))} />
                                </SmallCard>)}
                        </div>
                        <PrimaryButton className='mt-5' >حفظ</PrimaryButton>
                    </form>
                )}
            </Modal>
            {roles.map((role, i) => <div key={i}>
                <div className="flex items-end gap-3 border-b p-2 justify-between mb-3">
                    <h4 className='' >
                        {role.name}
                    </h4>
                    <div className="flex gap-2">
                        <ButtonEdit onClick={()=> handleUpdate(role)} />
                        <ButtonDelete />
                    </div>
                </div>
                <div className="flex ">
                    {role.permissions.map((item, i) =>
                        <SmallCard key={i}>
                            {item.name}
                        </SmallCard>)}
                </div>
            </div>)}
        </DashboardLayout>
    )
}

export default Roles
